import os
import tempfile
import shutil
from PIL import Image, ImageDraw, ImageFont
from mutagen.mp3 import MP3
from mutagen.id3 import ID3, TIT2, TCOM, COMM
import ffmpeg

# Watermark text to add to all files
watermark_text = [
    "Public Domain (CC0) Gift",
    "ORIGINAL ANTHRO OFFERING",
    "[ARCHIVAL TIMESTAMP]",
    "https://anthroentertainment.com/",
    "https://donate.anthroentertain.com/"
]

# Function to add watermark on images (PNG, JPG)
def watermark_image(image_path, temp_dir):
    with Image.open(image_path) as img:
        draw = ImageDraw.Draw(img)
        font = ImageFont.load_default()  # You can change this to a more professional font if needed

        margin = 20
        line_height = 20
        text_position_y = margin

        # Loop through each line of the watermark text and add it to the image
        for line in watermark_text:
            # Get text size using textbbox (bounding box of the text)
            bbox = draw.textbbox((0, 0), line, font=font)
            text_width, text_height = bbox[2] - bbox[0], bbox[3] - bbox[1]
            text_position_x = (img.width - text_width) // 2  # Center the text

            # Ensure we don't overwrite the content by adding vertical spacing
            if text_position_y + text_height > img.height - margin:
                break  # Stop if there's no space left

            # Draw the text
            draw.text((text_position_x, text_position_y), line, font=font, fill=(255, 255, 255))
            text_position_y += text_height + line_height

        # Save the image with the watermark to the temporary directory
        temp_image_path = os.path.join(temp_dir, os.path.basename(image_path))
        img.save(temp_image_path)
        print(f"Watermarked image saved: {temp_image_path}")

# Function to add metadata to audio files (MP3, WAV)
def watermark_audio(audio_path, temp_dir):
    audio = MP3(audio_path, ID3=ID3)
    
    # Avoid overlapping metadata tags by adding new unique fields
    audio.tags.add(TIT2(encoding=3, text="Public Domain (CC0) Gift"))
    audio.tags.add(TCOM(encoding=3, text="Original Anthro Offering"))
    audio.tags.add(COMM(encoding=3, lang='eng', desc='description', text="Timestamp and archival info"))
    
    # Save and overwrite the original file to the temporary directory
    temp_audio_path = os.path.join(temp_dir, os.path.basename(audio_path))
    audio.save(temp_audio_path)
    print(f"Watermarked audio saved: {temp_audio_path}")

# Function to add watermark to video (MP4)
def watermark_video(video_path, temp_dir):
    # Save the watermarked video to the temporary directory
    temp_video_path = os.path.join(temp_dir, os.path.basename(video_path))
    ffmpeg.input(video_path).output(temp_video_path, vf="drawtext=text='Public Domain (CC0) Gift':fontcolor=white:fontsize=24:x=(w-text_w)/2:y=10").run()
    print(f"Watermarked video saved: {temp_video_path}")

# Function to process all files in a folder recursively
def process_files_in_folder(folder_path):
    # Create a temporary directory
    with tempfile.TemporaryDirectory() as temp_dir:
        for root, dirs, files in os.walk(folder_path):
            for filename in files:
                file_path = os.path.join(root, filename)

                # Process image files (png, jpg)
                if filename.lower().endswith(('png', 'jpg', 'jpeg')):
                    watermark_image(file_path, temp_dir)

                # Process audio files (mp3, wav)
                elif filename.lower().endswith(('mp3', 'wav')):
                    watermark_audio(file_path, temp_dir)

                # Process video files (mp4)
                elif filename.lower().endswith(('mp4')):
                    watermark_video(file_path, temp_dir)

        # The temporary directory and all its contents are automatically deleted after the block

# Main execution
if __name__ == "__main__":
    # Automatically process from the current directory and all its subdirectories
    process_files_in_folder(os.getcwd())
